/**** A P P L I C A T I O N   N O T E   ************************************
*
* Title			: DMX512 reception library
* Version		: v2.5
* Last updated	: 29.11.06
* Target		: Botex, SP1500-DMX [ATmega8515]
* Clock			: 16MHz
*
* written by hendrik hoelscher, www.hoelscher-hi.de
***************************************************************************
 This program is free software; you can redistribute it and/or 
 modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation; either version2 of 
 the License, or (at your option) any later version. 

 This program is distributed in the hope that it will be useful, 
 but WITHOUT ANY WARRANTY; without even the implied warranty of 
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 General Public License for more details. 

 If you have no copy of the GNU General Public License, write to the 
 Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 

 For other license models, please contact the author.

;***************************************************************************/


.include "m8515def.inc"

;Register and Pin definitions

.def TempA = r16
.def TempB= r17
.def TempC= r18
.def TempD= r3
.def MainFlags= r19
.def ErrFlags=r20
.def StroboHeatH=r21
.def StroboHeatL=r22
.def HeatInc= r9
.def DmxState= r23
.def TCNT0H = r8
.def null= r2
.def SREG_buf= r10

.equ LED= 		 PB3
.equ CNTRL_LED=  PD7
.equ Strobo= 	 PB7


.equ ONTIME= 800	 ;Time the strobe is triggered

;** vars in SRAM 

.equ SPEEDraw=   0x61 ;Buffer for Speed
.equ BRIGHTraw=  0x62 ;Buffer for Brightness
.equ MODEraw=    0x63 ;Buffer for strobe mode

.equ SPEEDclcH=  0x64 ;Buffer for Speed
.equ SPEEDclcL=  0x65
.equ FIREclcH=   0x66 ;Buffer for Brightness
.equ FIREclcL=   0x67 ;Buffer for Brightness
.equ QUENCHclcH= 0x68 ;Buffer for Brightness
.equ QUENCHclcL= 0x69 ;Buffer for Brightness
.equ MODEclc=    0x6A ;Buffer for strobe mode

.equ ERRCODE=    0x6B ;error LED pattern
.equ ERRprescaler= 0x6D ;prescaler for error indicator
.equ RANDSEED=	 0x70


;***** special flags

.equ  OVERHEAT		=0		;Error Flags
.equ  VALID_DMX		=1
.equ  SIGNAL_COMING =2
.equ  MANUAL		=4		;use THIS for checking current mode!
.equ  XOP_OFF		=5
.equ  ZChigh		=6		;for old board's polling
.equ  REV_5_2		=7

.equ  doSPEED		=0		;Main Flags
.equ  doBRIGHT      =1
.equ  doMODE		=2
.equ  doIndicate	=3
.equ  wasLOW		=4  	;used for single flash	
.equ  DMXisSTANDBY	=5
.equ  isREADY		=6
.equ  doFLASH		=7

.equ  NORMAL_MODE	=0
.equ  BLINDING_MODE =1
.equ  RANDOM_MODE   =2
.equ  SINGLE_FLASH  =3
.equ  SYNC_MODE		=4		

;interrupt vectors

.org 0
	
	rjmp reset 				;reset vector address
	rjmp zero_crossing		;External Interrupt0 Vector Address
	reti					;External Interrupt1 Vector Address
	reti					;Input Capture1 Interrupt Vector Address
	rjmp fire				;Output Compare1A Interrupt Vector Address
	rjmp quench				;Output Compare1B Interrupt Vector Address
	reti					;Overflow1 Interrupt Vector Address
	rjmp dmx_flash 			;Overflow0 Interrupt Vector Address
	reti					;SPI Interrupt Vector Address
	rjmp get_byte			;UART Receive Complete Interrupt Vector Address
	reti					;UART Data Register Empty Interrupt Vector Address
	reti					;UART Transmit Complete Interrupt Vector Address
	reti					;Analog Comparator Interrupt Vector Address
	reti					;External Interrupt2 Vector Address
	reti 					;Output Compare0 Interrupt Vector Address
	reti					;EEPROM Interrupt Vector Address
	reti					;SPM complete Interrupt Vector Address
	reti					;SPM complete Interrupt Vector Address	

;INIT
reset:
cli

;Stack
	ldi		TempA, high(RAMEND)
	out		SPH,TempA
	ldi		TempA, low(RAMEND)
	out		SPL,TempA

;***** WATCHDOG
	wdr
	ldi 	tempA, (1<<WDE)|(1<<WDCE)
	out 	WDTCR, tempA
	ldi 	tempA, (1<<WDE)|(1<<WDCE)|(1<<WDP2)|(1<<WDP1)|(1<<WDP0) ;set TimeOut to 2.1s
	out 	WDTCR, tempA
	wdr

;***** set Ports

; PortA
	clr 	TempA
	out		DDRA, TempA
	ser		TempA
	out		PortA, TempA		;DIP-Switches with pullup

; PortB
	ldi		TempA, 0b11001000 	;0,1=invers_chck 3=LED 4=ZC 6,7=OUT
	out		DDRB, TempA
	ldi		TempA, 0b11111011
	out		PortB, TempA

; PortC
	ldi		TempA, 0b00000000 	;0,1=DIP-switches
	out		DDRC, TempA
	ldi		TempA, 0b00000011
	out		PortC, TempA

; PortD
	ldi		TempA, 0b10000000 	;Cntrl-LED
	out		DDRD, TempA
	ldi		TempA, 0b10000100
	out		PortD, TempA

; PortE
	;*unused*
	
	clr		ErrFlags

  EEread_target:				;get target board
	sbic	EECR, EEWE
	rjmp	EEread_target
	clr		TempA
	out		EEARH, TempA
	ldi		TempA, 2
	out		EEARL, TempA
	sbi		EECR, EERE
	in		TempA, EEDR
		
	cpi		TempA, 1
	breq	init_trgt			;old revision
	cpi		TempA, 2
	breq	new_board			;new revision
	rjmp	get_target
   new_board:
   	sbr		ErrFlags, (1<<REV_5_2)
   init_trgt:

; Timers	
	clr		TempA
	out		TCCR1A, TempA
	ldi 	TempA, (1<<CS11)	;set counter1 @clock/8 frequency
	out 	TCCR1B, TempA
	ldi 	TempA, (1<<CS02)	;set counter0 @clock/256 frequency
	out 	TCCR0, TempA				
	ldi 	TempA, (1<<TOIE0) 	;enable Timer0 overflow interrupt
	out 	TIMSK, TempA

; SPI
	clr		TempA
	out		SPCR, TempA			;disable SPI

;ext interrupt 0 for phase zero-crossing control (just new boards!)
	sbrs 	ErrFlags, REV_5_2
	rjmp 	no_ext_int			;old board
	
	ldi 	TempA, (1<<ISC01)	;falling edge
	out 	MCUCR, TempA
	ldi		TempA, (1<<INT0)	;activate
	out 	GICR, TempA 		;		
	ldi		TempA, (1<<INTF0)	;clear interrupt0 flag
	out 	GIFR,  TempA

no_ext_int:
		
;USART
	ldi 	TempA, 0x03			;set USART to 250 kbaud @ 16Mhz
   	out 	UBRRL, TempA
   	ldi 	TempA, 0x00
   	out 	UBRRH, TempA
   	ldi 	TempA, (1<<URSEL)|(3<<UCSZ0)|(1<<USBS) ; 8n2
   	out 	UCSRC, TempA			;
	in		TempA, UDR			;clear Uart Receive Interrupt Flag
	clr		TempA
 	out		UCSRA, TempA		;clear all error flags, disable U2X and MPCM
	clr     DmxState

; MODE SELECT		
	rcall	get_opmode
	sbrc	ErrFlags, MANUAL
	rjmp	init_1				;don't init in manual mode
	ldi		TempA, (1<<RXCIE)|(1<<RXEN) ;enable Uart RX IRQ & Uart RXD in DMX mode
	out		UCSRB, TempA
   init_1:	
	cbi     PortB, LED
	
; set Register
	ldi 	MainFlags, (1<<DMXisSTANDBY)|(1<<isREADY)|(1<<doSPEED)|(1<<doBRIGHT)|(1<<doMODE)	
	ldi 	StroboHeatL, 5
	clr		StroboHeatH
	clr 	HeatInc
	clr 	null
	sts		SPEEDraw, null
	sts		BRIGHTraw, null
	sts		MODEraw,  null
	ldi 	TempA, 3
	sts		ERRcode, TempA
	ldi		TempA, 4
	sts		ERRprescaler, TempA

	wdr

	wdr									;Watchdog-Reset
	sei

;****************************** main (calculations) *****************************

main:
		sbrc	MainFlags, doSPEED		;should calc new speed?
		rjmp	calc_speed
	 	sbrc	MainFlags, doBRIGHT		;should calc new brightness?
	 	rjmp	calc_brightness
		sbrc	MainFlags, doMODE		;should calc new mode?
		rjmp	calc_mode
		sbrc	MainFlags, doINDICATE	;error indicator and heat cntrl (called by zc @10Hz)
		rjmp	indicate

		sbrc	ErrFlags, REV_5_2
		rjmp	main					;new boards need no polling
		
		;detect ZC !
		cli
		sbis	PinB,4					;ZC high?
		rjmp	check_zc				;no
		sbr		ErrFlags, (1<<ZChigh)	;yes  ->neg. half wave
		rjmp	exit_poll
	   check_zc:
	  	sbrs 	ErrFlags, ZChigh		;was high before and now low -> zc!!
		rjmp	exit_poll				;no (pos. half wave)
		cbr		ErrFlags, (1<<ZChigh)	;wait for next neg. half wave
		rcall   zero_crossing			;ZC detected!!
		rjmp	main		
	   exit_poll:
		sei
		rjmp	main


;____________________________________________________________________________________

	calc_speed:
		cbr		MainFlags, (1<<doSPEED)

		lds		TempA,  MODEclc
		cpi		TempA,  SINGLE_FLASH
		breq	create_flash
		cpi		TempA,  BLINDING_MODE
		breq	blinding_speed
		cpi		TempA,  RANDOM_MODE
		breq	random_speed


   normal_speed:  
		lds		TempA,	SPEEDraw

		cpi		TempA, 5				;val <= 5 -> stand by
		brsh	spd_1
		sbr		MainFlags, (1<<DMXisSTANDBY)
		rjmp	main
	   spd_1:
	   	subi	TempA, 5				;range: 0-250 (1-14Hz)
		ldi		ZH, high(flash_rate*2)
		ldi		ZL, low(flash_rate*2)
		add		ZL, TempA
		adc		ZH, null
		add		ZL, TempA
		adc		ZH, null
		lpm		TempA,  Z+				;get flash rate
		lpm		TempB, Z
     spd_finish:
	 	inc		TempB
		neg		TempA					;clc T0 offset
		cli								;atomic operation
		sts		SPEEDclcL, TempA
		sts		SPEEDclcH, TempB
		cp		TempB, TCNT0H			;soft speed blending
		brsh	spd_f1
		mov		TCNT0H, TempB			;atomic write	
		out		TCNT0, TempA
	   spd_f1: 
	    sei
		cbr		MainFlags, (1<<DMXisSTANDBY) 	
		rjmp	main

	
	create_flash:
		lds		TempA,  SPEEDraw
		cpi		TempA,  128
		brsh	rq_flash
		sbr		MainFlags, (1<<wasLOW)
		rjmp	main
	   rq_flash:
	   	sbrs	MainFlags, wasLOW		;flash only, when ch was low before!
		rjmp	main
		cbr		MainFlags, (1<<wasLOW)
		sbr		MainFlags, (1<<doFlash)
		rjmp	main

	
	random_speed:								;use SPEEDraw in TimerISR 
		lds		TempA,	SPEEDraw
		cpi		TempA, 5						;val <= 5 -> stand by
		brsh	rnd_1
		sbr		MainFlags, (1<<DMXisSTANDBY)
		rjmp	main
       rnd_1:
		ldi		TempB,  high(3700)
		ldi		TempA, low(3700)				;use ~16.8Hz as base freq in random mode
		rjmp	spd_finish


	blinding_speed:
		lds		TempA,  SPEEDraw
		cpi		TempA,  64
		brsh	bs_1
		sbr		MainFlags, (1<<DMXisSTANDBY)	;shut down
		rjmp	main
	   bs_1:
	    cpi		TempA,  96
		brsh	bs_2
		ldi		TempB,  high(4464)				;14Hz
		ldi		TempA, low(4464)
		rjmp	spd_finish	
		
	   bs_2:
	    cpi		TempA,  128
		brsh	bs_3
		ldi		TempB,  high(4167)
		ldi		TempA, low(4167)				;15Hz
		rjmp	spd_finish
		
	   bs_3:
	    cpi		TempA,  160
		brsh	bs_4
		ldi		TempB,  high(3906)
		ldi		TempA, low(3906)				;16.6Hz
		rjmp	spd_finish
		
	   bs_4:
	    cpi		TempA,  192
		brsh	bs_5
		ldi		TempB,  high(3125)
		ldi		TempA, low(3125)				;20Hz
		rjmp	spd_finish
		
	   bs_5:
	    cpi		TempA,  224
		brsh	bs_6
	   	ldi		TempB,  high(2500)
		ldi		TempA, low(2500)				;25Hz
		rjmp	spd_finish
	    
       bs_6:
		ldi		TempB,  high(1250)
		ldi		TempA, low(1250)				;50Hz
		rjmp	spd_finish
	    
;___________________________________________________________________________________________________
										
	calc_brightness:
		cbr		MainFlags, (1<<doBRIGHT)
		rcall	calc_heatInc			;calc new inc val. for heat simulation

		lds		TempA, BRIGHTraw
		cpi		TempA, 5				;val <= 5 -> stand by
		brsh	brt_1
		sbr		ErrFlags, (1<<XOP_OFF)	;standby if <5
		rjmp	main
	   brt_1:
		cbr		ErrFlags, (1<<XOP_OFF)
	   	ldi		ZH, high(firing_curve*2);set pointer to base address
		ldi		ZL, low(firing_curve*2)
		add		ZL, TempA
		adc		ZH, null
		add		ZL, TempA
		adc		ZH, null
		lpm		TempA, Z+				;load brightness
		lpm		TempB, Z			

		cli								;atomic operation
		sts		FIREclcL, TempA			
		sts		FIREclcH, TempB

		ldi		TempC, high(ONTIME)		;calc quench time
		mov		TempD, TempC
		ldi		TempC, low(ONTIME)
		add		TempA, TempC
		adc		TempB, TempD
		sts		QUENCHclcL, TempA
		sts		QUENCHclcH, TempB
		sei	
		rjmp 	main


	calc_heatInc:
		lds		TempA, BRIGHTraw
		swap	TempA
		lsr		TempA
		lsr		TempA
		andi	TempA, 0b00000011
		inc		TempA
		inc		TempA
		mov     HeatInc,	TempA
		ret

;_______________________________________________________________________________

	calc_mode:
		cbr		MainFlags, (1<<doMODE)

		lds		TempA, MODEraw
		cpi		TempA, 200
		brsh	synchronize
		cpi		TempA, 150
		brsh	single
		cpi		TempA, 100
		brsh	random
		cpi		TempA, 50
		brsh	blinder
		ldi		TempB, NORMAL_MODE				;0-63 -> normal mode
		rjmp	mode_finish
	   random:
		sts		RANDSEED, TempA					;use raw mode val as new random seed
	    ldi		TempB, RANDOM_MODE
		rjmp	mode_finish
	   blinder:
	    ldi		TempB, BLINDING_MODE			;64-127 -> Blinder
		rjmp	mode_finish
	   single:
	    ldi		TempB, SINGLE_FLASH				;128-190 -> single flash
		sbr		MainFlags, (1<<DMXisSTANDBY)	;disable T0-speed
		rjmp	mode_finish
	   synchronize:								;192-255 -> sync multiple Strobos
		ldi		TempB, SYNC_MODE

	   mode_finish:
	    lds		TempA, MODEclc
		cp		TempA, TempB
		breq	mode_exit
	    sts		MODEclc, TempB
		cpi		TempB, SYNC_MODE
		breq	mode_sync
		sbr		MainFlags, (1<<doSPEED)			;eval new speeds
	   mode_exit:
		rjmp	main
		
       mode_sync:
		ldi		TempA, 1
		cli
		mov		TCNT0H,TempA					;force instant flash (and get in defined state)
		ser		TempA
		out		TCNT0, TempA
		sei
		rjmp	main

	
;_______________________________________________________________________________

	indicate:
		cbr		MainFlags, (1<<doINDICATE)
		
		ldi		TempA, 200						;check restart point
		cp		StroboHeatL, TempA
		cpc		StroboHeatH, null
		brlo	ind_go
		cpi		StroboHeatH, 2					;check shut down point
		brsh	ind_stop
	  ind_ret:
	  	ldi		TempA, 3
	  	cp		StroboHeatL, TempA
		cpc		StroboHeatH, null
		brlo	ind_1						
		sub		StroboHeatL, TempA
		sbc		StroboHeatH, null
				
	   ind_1:
	   ;ErrorLED
		lds	  	TempA, ERRcode
	   	
		sbi   	PortB, LED
		sbrc  	TempA, 0						;if 1st bit HI
	 	cbi  	PortB, LED

	    lsr   	TempA
		cpi		TempA, 1
		brne	no_ind	 
			
        ;wenn durchrotiert (ERRpos = 0)
		ldi		TempA, 0b10000000				;load empty pattern
		sbic	UCSRB, RXEN 					;if manual mode -> no DMX indication		
		sbrc 	ErrFlags, VALID_DMX
		rjmp	dmx_ok
		ldi	 	TempA, 0b10000101				;load specific error codes
		sbrs	ErrFlags, SIGNAL_COMING
		ldi	 	TempA, 0b10000001
		rjmp	dmx_nok
	   dmx_ok:
	    wdr										;reset watchdog
	   dmx_nok:
		sbrc	ErrFlags, OVERHEAT
		ser		TempA
		cbr  	ErrFlags, (1<<VALID_DMX)|(1<<SIGNAL_COMING)

	   no_ind:
		sts		ERRcode, TempA
	   	rjmp	main						

	   ind_go:						
	   	sbrc	ErrFlags, OVERHEAT
		cbr		MainFlags, (1<<doFlash)			;cancel flash request if restart
		cbr		ErrFlags, (1<<OVERHEAT) 		;strobo is ready again
		rjmp	ind_ret
	   ind_stop:
	   	sbr		ErrFlags, (1<<OVERHEAT) 		;strobo is too hot -> shut down
		rjmp	ind_ret

;________________________________________________________________________________

	global_mode:
		;MANUAL/DMX-SELECT
	    rcall	get_opmode						;get current operating mode
		sbrc	ErrFlags, MANUAL				;manual mode selected?
		rjmp	man_mode						; yes
		sbic	UCSRB, RXEN	 					;new in DMX mode?
		ret										; no
		ldi		TempA, (1<<RXCIE)|(1<<RXEN) 	;enable Uart RX IRQ & Uart RXD
		out		UCSRB, TempA
		ret
				
	  man_mode:
	  	out		UCSRB, null						;disable USART

		;CHOOSE PARAMETERS
		rcall	get_address
		lds		TempB, MODEraw
		cp		TempB, XL	 					;have DIP settings changed?
		breq	exit_man		 				;no...
		sts		MODEraw, XL						;save new val
		mov		TempB, XL

;	(->Speed)
		andi	TempB, 0b00001111				;mask lower nibble 

		cpi		TempB, 0
		breq	man_single_flash
		cpi		TempB, 15
		breq	man_blind
		inc		TempB
		push	TempC
		ldi		TempC, 16
		mul		TempB, TempC					;expand val
		pop		TempC
		mov		TempB, r0
		subi	TempB, -10						;max speed
		sts		SPEEDraw, TempB					;store new val	
		sbr		MainFlags, (1<<doSpeed) 		;calc new speed in main
		rjmp	dimm_man
		
	   man_single_flash:
	    sbr		MainFlags, (1<<doFlash)|(1<<DMXisStandBy) ;request flash
		rjmp	dimm_man

	   man_blind:
	   	ldi		TempB,  (high(2500) +1)
		mov		TCNT0H, TempB
		ldi		TempB, low(2500)				;25Hz
		neg		TempB							;clc T0 offset
		cli
		sts		SPEEDclcL, TempB
		sts		SPEEDclcH, TCNT0H	
		out		TCNT0, null 
		sei
		cbr		MainFlags, (1<<DMXisSTANDBY)	
				
;   (->Brightness)
	   dimm_man:
	    andi	XL, 0b11110000					;mask higher nibble
		subi	XL, -10							;prevent black out 
		sts		BRIGHTraw, XL	 			 	;store new val	
		sbr		MainFlags, (1<<doBright) 		;calc new brightness in main
		
	   exit_man:
		ret	


;****************************** Fire Timing *************************************

fire:				
		cbi		PortB, 	Strobo				;FIRE!
		reti


quench:
		in      SREG_buf, SREG
		push    TempA

		sbi		PortB, 	Strobo				;quench

		in		TempA,   TIMSK				;disable interrupt
		cbr		TempA,   (1<<OCIE1A)|(1<<OCIE1B)
		out		TIMSK,	 TempA	
   
		sbr		MainFlags, (1<<isREADY)		;Ready...
	
		pop		TempA
		out     SREG, SREG_buf	
		reti


;****************************** zero crossing ***********************************

zero_crossing:
		in      SREG_buf, SREG
		push	TempA
		push	TempB

		sbrs	MainFlags, doFLASH			;flash request?
		rjmp	exit_zc 					; no...
		;set up new flash
		sbrc	ErrFlags, OVERHEAT 			;is strobo too hot?
		rjmp	exit_zc						; no...
		sbrs	MainFlags, isREADY 			;is strobo ready to fire?
		rjmp	exit_zc						; no...

		cbr		MainFlags, (1<<doFlash)|(1<<isREADY) ;quit flags

		add		StroboHeatL, HeatInc  		;increase heat!
		adc		StroboHeatH, null

		lds		TempB, 	 FIREclcH			;set firing angle
		lds		TempA,   FIREclcL
		out		OCR1AH,  TempB	
		out		OCR1AL,  TempA

		lds		TempB, 	 QUENCHclcH			;set quench angle
		lds		TempA,   QUENCHclcL
		out		OCR1BH,  TempB
		out		OCR1BL,  TempA

		out		TCNT1H,  null				;clear counter
		out		TCNT1L,  null

		in		TempB,   TIFR				;clear irq flags
		sbr		TempB,   (1<<OCF1A)|(1<<OCF1B)
		out		TIFR,    TempB

		in		TempB,   TIMSK				;enable interrupts
		sbr		TempB,   (1<<OCIE1A)|(1<<OCIE1B)
		out		TIMSK,	 TempB

	exit_zc:
		lds		TempB,	 ERRprescaler
		dec		TempB
		brne	zc_finish
		sbr		MainFlags, (1<<doINDICATE)	;error indicator and heat cntrl
		ldi		TempB, 4					;10Hz 
		zc_finish:
		sts		ERRprescaler, TempB
		rcall	global_mode					;has global mode changed?
		pop		TempB
		pop		TempA
		out     SREG, SREG_buf	
		reti

;****************************** DMX-Flashrate ***********************************
dmx_flash:
		in      SREG_buf, SREG
		push	ZH
		push    ZL
		push	TempB
	
		dec		TCNT0H					; decrement counter extension
		brne	exit_dflash
		
		lds		TempB, SPEEDclcL		; set nxt flash time
		out		TCNT0, TempB
		lds		TCNT0H, SPEEDclcH
		
		sbrc	MainFlags, DMXisSTANDBY	; if standby -> no flash
		rjmp	exit_dflash
		
		lds		TempB, MODEclc
		cpi		TempB, RANDOM_MODE
		brne	req_dflash				; request flash if not in random mode
		
		
		ldi		ZH, high(random_table*2);set pointer to base address
		lds		ZL, RANDSEED
		lpm		TempB, Z+				;load brightness
		sts		RANDSEED, ZL
		
		lds		ZL, SPEEDraw	
		lsr		ZL		
		cp		ZL, TempB
		brlo	exit_dflash	

    req_dflash:
		sbrs	ErrFlags, XOP_OFF
		sbr		MainFlags, (1<<doFlash)	; flash request

	exit_dflash:
		pop		TempB
		pop		ZL
		pop		ZH
		out     SREG, SREG_buf	
		reti

		
;****************************** Empfang *****************************************

get_byte:
		in      SREG_buf, SREG
		push    TempA
		push	TempB
		
		in		TempA, UCSRA
		in		TempB, UDR
		
		sbr 	ErrFlags, (1<<SIGNAL_COMING);message for indicator...		

		sbrc	TempA,DOR				;check for overrun -> wait for reset
		rjmp	overrun	

		sbrc	TempA, FE				;check 1st stop bit
		rjmp	gb_break
		
		cpi		DmxState, 1     		;check for startbyte (else frame error -> reset)
		breq	startbyte	

		cpi 	DmxState, 2     		;check for start adress
		breq    startadr
  
		cpi 	DmxState, 3	    		
  		breq	get_speed

		cpi		DmxState, 4
		breq	get_brightness

		cpi		DmxState, 5
		breq	get_mode

		clr		DmxState			 	;wait for break
	exit_dmx_receive:	
		pop		TempB
		pop		TempA
		out     SREG, SREG_buf	
		reti		

	startbyte:
		tst 	TempB	 				;if null, then Startbyte
		brne 	overrun
		
		inc  	DmxState 				;next -> start address
		
		rcall	get_address				;read start address
	   
		cp		XL, null				;Startadresse = 0 is nich!
		cpc     XH, null
		brne	exit_dmx_receive	
	
	    clr		DmxState		 		;quit receive routine
		sts		BRIGHTraw, null		    ;set brightness to null
		sbr		MainFlags, (1<<doBRIGHT) ;calc new speed in main		
	    rjmp	exit_dmx_receive

 	startadr:
		sbiw	XH:XL, 1
		brne	exit_dmx_receive
		inc 	DmxState		 		;next-> Byte-Evaluation

	get_speed:
		inc 	DmxState				;next channel
		lds     TempA, SPEEDraw  		;load saved speed
		cp		TempA, TempB	 		;has changed?
		breq 	exit_dmx_receive
		sts		SPEEDraw, TempB  		;store new val	
		sbr		MainFlags, (1<<doSpeed) ;calc new speed in main	
		rjmp 	exit_dmx_receive
	
	get_brightness:
		inc 	DmxState				;next channel
		lds     TempA, BRIGHTraw  		;load saved brightness
		cp		TempA, TempB	 		;has changed?
		breq	exit_dmx_receive
		sts		BRIGHTraw, TempB 		;store new val	
		sbr		MainFlags, (1<<doBRIGHT) ;calc new brightness in main
		rjmp 	exit_dmx_receive

	get_mode:
		clr		DmxState
		sbr 	ErrFlags, (1<<VALID_DMX);message for indicator...
		lds		TempA, MODEraw
		cp		TempA, TempB
		breq	exit_dmx_receive
		sts		MODEraw, TempB  		;store new val
		sbr		MainFlags, (1<<doMODE) 	;calc new mode in main
		rjmp 	exit_dmx_receive
		
	gb_break:
		cbi		UCSRA,FE				;clear frame-error flag
		tst		TempB					;byte must be zero (88s LO)
		brne	gb_break1
	 	ldi 	DmxState, 1				;next -> control of start byte
	   gb_break1:
		rjmp 	exit_dmx_receive

	overrun:
	 	clr 	DmxState				;wait for framing error
		cbi		UCSRA,DOR				;clear overrun flag !!
		rjmp 	exit_dmx_receive
		

;****************************** DIP states ****************************************
get_address:
		cli								;atomic operation
		sbic	PinB, PB0				;this bridge is open in inverted strobes!
		rjmp	ga_invert				;get inverted start address
		in 		XL, PinA				;Einlesen der Startadresse		
		com		XL						;Invertierung
		clr		XH
		sbis	PinC, PC0				;9.address bit
		inc		XH
		reti
	  ga_invert:
	 	clr		XH
		clr		XL
		sbis	PinA,PA1				;9th bit
		inc		XH
		sbis	PinA,PA2
		sbr		XL, (1<<7)
		sbis	PinA,PA3
		sbr		XL, (1<<6)
		sbis	PinA,PA4
		sbr		XL, (1<<5)
		sbis	PinA,PA5
		sbr		XL, (1<<4)
		sbis	PinA,PA6
		sbr		XL, (1<<3)
		sbis	PinA,PA7
		sbr		XL, (1<<2)
		sbis	PinC,PC0
		sbr		XL, (1<<1)
		sbis	PinC,PC1
		sbr		XL, (1<<0)		
	 	reti


get_opmode:								;operating mode
		cli								;atomic operation
		cbr		ErrFlags, (1<<MANUAL)	
		sbic	PinB, PB0				;this bridge is open in inverted strobes!
		rjmp	gopm_invert				;get inverted start address
		sbis	PinC, PC1
		sbr		ErrFlags, (1<<MANUAL)
		reti
      gopm_invert:
	  	sbis	PinA, PA0
		sbr		ErrFlags, (1<<MANUAL)
		reti
	  	


;****************************** check Revision *****************************************

.equ poll_HI  =0;
.equ irq_HI   =1;

get_target:
		clr  	R20
		ldi  	R18, 50		
		ldi  	R19, 50

	gt_loop:		 		
 		ldi  	R16, $FF					; make 16Hz flashing
  trgt1:ldi  	R17, $C4
  trgt2:  								; chck zc-source
       	sbis 	PinB, 4					; Poll pin HI?
		rjmp 	poll_low
		sbr  	R20, (1<<poll_HI)
       ret_poll:
		sbis 	PinD, 2					; irq pin HI?
		rjmp 	irq_low
		sbr  	R20, (1<<irq_HI)
	   ret_irq:
    	dec  	R17
        brne 	trgt2
		wdr
        dec  	R16
        brne 	trgt1
          

		sbis 	PortB,LED				;blink
		rjmp 	trgt_off
		cbi		PortB,LED
		cbi		PortD,CNTRL_LED
		rjmp 	gt_loop
	   trgt_off:
		sbi 	PortB,LED
		sbi		PortD,CNTRL_LED
		rjmp	gt_loop
		
		poll_low:
	    sbrs	R20, poll_HI
		rjmp	ret_poll
		cbr		R20, (1<<poll_HI)	; zc happened
		dec		R18					; dec poll zc counter
		brne	ret_poll
		ldi		TempA, 1
		rjmp	TARGETwrite			; old board detected

		irq_low:
		sbrs	R20, irq_HI
		rjmp	ret_irq
		cbr		R20, (1<<irq_HI)	; zc happened
		dec		R19					; dec irq zc counter
		brne	ret_irq
		ldi		TempA, 2			; new board detected

	   TARGETwrite:
		sbic	EECR, EEWE
		rjmp	TARGETwrite
		clr		TempB
		out		EEARH, TempB
		ldi		TempB, 2
		out		EEARL, TempB
		out		EEDR, TempA
		sbi		EECR, EEMWE
		sbi		EECR, EEWE
		
		rjmp	reset

	   


nix: rjmp nix


firing_curve:                              ;firing table for linearization of brightness
.dw 17963, 17625, 17395, 17215, 17066, 16936, 16821, 16716, 16621, 16533, 16450, 16372, 16299, 16230, 16163, 16100 
.dw 16039, 15981, 15924, 15870, 15817, 15766, 15717, 15668, 15621, 15576, 15531, 15487, 15445, 15403, 15362, 15322 
.dw 15283, 15244, 15206, 15169, 15132, 15096, 15060, 15025, 14991, 14957, 14923, 14890, 14858, 14825, 14793, 14762 
.dw 14731, 14700, 14670, 14640, 14610, 14580, 14551, 14522, 14494, 14465, 14437, 14409, 14382, 14354, 14327, 14300 
.dw 14273, 14247, 14220, 14194, 14168, 14142, 14117, 14091, 14066, 14041, 14016, 13991, 13966, 13942, 13918, 13893 
.dw 13869, 13845, 13821, 13798, 13774, 13750, 13727, 13704, 13680, 13657, 13634, 13611, 13589, 13566, 13543, 13521 
.dw 13498, 13476, 13454, 13431, 13409, 13387, 13365, 13343, 13321, 13299, 13278, 13256, 13234, 13213, 13191, 13170 
.dw 13148, 13127, 13106, 13084, 13063, 13042, 13021, 13000, 12979, 12957, 12936, 12915, 12895, 12874, 12853, 12832 
.dw 12811, 12790, 12769, 12749, 12728, 12707, 12686, 12666, 12645, 12624, 12603, 12583, 12562, 12541, 12521, 12500 
.dw 12479, 12459, 12438, 12417, 12397, 12376, 12355, 12334, 12314, 12293, 12272, 12251, 12231, 12210, 12189, 12168 
.dw 12147, 12126, 12105, 12084, 12063, 12042, 12021, 12000, 11979, 11958, 11937, 11915, 11894, 11873, 11851, 11830 
.dw 11808, 11787, 11765, 11743, 11721, 11700, 11678, 11656, 11633, 11611, 11589, 11567, 11544, 11522, 11499, 11476 
.dw 11454, 11431, 11408, 11385, 11361, 11338, 11314, 11291, 11267, 11243, 11219, 11195, 11170, 11146, 11121, 11096 
.dw 11071, 11046, 11020, 10995, 10969, 10943, 10916, 10890, 10863, 10836, 10808, 10781, 10753, 10724, 10696, 10667 
.dw 10637, 10608, 10577, 10547, 10516, 10484, 10452, 10419, 10386, 10352, 10317, 10282, 10246, 10209, 10171, 10132 
.dw 10092, 10050, 10007, 9963,  9917,  9869,  9818,  9765,  9709,  9648,  9582,  9510,  9428,  9330,  9204,  8900

flash_rate:                              ;table for speeds in normal mode
.dw 64592, 60094, 56479, 53480, 50932, 48729, 46794, 45076, 43534, 42139, 40868, 39703, 38629, 37634, 36708, 35844 
.dw 35034, 34272, 33555, 32877, 32234, 31624, 31044, 30492, 29964, 29459, 28976, 28513, 28068, 27640, 27229, 26832 
.dw 26449, 26080, 25723, 25378, 25044, 24720, 24406, 24102, 23806, 23519, 23240, 22968, 22704, 22446, 22195, 21951 
.dw 21712, 21480, 21252, 21031, 20814, 20602, 20395, 20192, 19994, 19800, 19610, 19423, 19241, 19062, 18887, 18715 
.dw 18546, 18380, 18218, 18058, 17901, 17747, 17596, 17447, 17301, 17158, 17016, 16877, 16740, 16606, 16473, 16343 
.dw 16214, 16088, 15963, 15840, 15720, 15600, 15483, 15367, 15253, 15140, 15029, 14920, 14812, 14705, 14600, 14496 
.dw 14393, 14292, 14192, 14093, 13996, 13900, 13805, 13711, 13618, 13526, 13436, 13346, 13257, 13170, 13083, 12998 
.dw 12913, 12830, 12747, 12665, 12584, 12504, 12425, 12346, 12269, 12192, 12116, 12041, 11966, 11892, 11819, 11747 
.dw 11675, 11605, 11534, 11465, 11396, 11328, 11260, 11193, 11127, 11061, 10996, 10932, 10868, 10804, 10741, 10679 
.dw 10618, 10556, 10496, 10436, 10376, 10317, 10258, 10200, 10142, 10085, 10029, 9972,  9917,  9861,  9806,  9752 
.dw 9698,  9644,  9591,  9538,  9486,  9434,  9382,  9331,  9280,  9230,  9180,  9130,  9081,  9032,  8983,  8935
.dw 8887,  8840,  8793,  8746,  8699,  8653,  8607,  8561,  8516,  8471,  8427,  8382,  8338,  8295,  8251,  8208 
.dw 8165,  8122,  8080,  8038,  7996,  7955,  7914,  7873,  7832,  7791,  7751,  7711,  7672,  7632,  7593,  7554 
.dw 7515,  7477,  7438,  7400,  7363,  7325,  7288,  7251,  7214,  7177,  7140,  7104,  7068,  7032,  6997,  6961 
.dw 6926,  6891,  6856,  6821,  6787,  6753,  6719,  6685,  6651,  6617,  6584,  6551,  6518,  6485,  6453,  6420 
.dw 6388,  6356,  6324,  6292,  6260,  6229,  6198,  6167,  6136,  6105,  6074,  6044

.org 0x500
random_table:
.db 0x18,   0xC7,   0x0B,   0xF6,   0xFF,   0xB0,   0xAF,   0x2E 
.db 0x2D,   0xF7,   0xB1,   0xE2,   0x20,   0x26,   0x45,   0x08 
.db 0x53,   0xE6,   0x3D,   0xB3,   0xD4,   0x16,   0xD1,   0xCF 
.db 0x34,   0x1B,   0x9F,   0x13,   0x76,   0xF0,   0x31,   0x4C 
.db 0x9B,   0x3F,   0x50,   0x01,   0xF4,   0xBB,   0x6D,   0x75 
.db 0xDA,   0x69,   0xB4,   0x60,   0x63,   0x3C,   0x3A,   0x12 
.db 0xCA,   0xD2,   0x55,   0x6B,   0xFD,   0xB2,   0x7D,   0xD5 
.db 0x57,   0x27,   0xE7,   0x82,   0xC9,   0x9D,   0x58,   0xF5 
.db 0x5F,   0x4D,   0x38,   0xC5,   0xE4,   0x1D,   0xE3,   0xC1 
.db 0x54,   0x89,   0x8B,   0x0E,   0xFB,   0x05,   0xA7,   0xF3 
.db 0x1F,   0x62,   0x7E,   0x2B,   0xA6,   0x39,   0x0C,   0xE8 
.db 0xAB,   0x11,   0x03,   0x36,   0x72,   0x49,   0xBF,   0x41 
.db 0x59,   0xDD,   0xDB,   0x6A,   0x98,   0x30,   0xF1,   0x42 
.db 0x52,   0x99,   0x61,   0xA4,   0xBD,   0xB5,   0x2F,   0x0F 
.db 0x33,   0xF9,   0x28,   0x92,   0x77,   0x90,   0x3B,   0x04 
.db 0xD9,   0x15,   0xB8,   0x44,   0xE9,   0x7F,   0xDF,   0xD3 
.db 0x94,   0x81,   0x6E,   0x24,   0xAC,   0xFC,   0x2C,   0xA9 
.db 0x46,   0xDE,   0x65,   0x19,   0x9E,   0x4F,   0xA1,   0x8C 
.db 0xD6,   0xEE,   0xB7,   0xC6,   0x95,   0x85,   0x79,   0x96 
.db 0x43,   0x29,   0x3E,   0x86,   0x5D,   0xFE,   0x51,   0x7A 
.db 0xEC,   0x0A,   0xC3,   0x67,   0xC2,   0xEF,   0xE0,   0x2A 
.db 0x6F,   0xB6,   0x91,   0x47,   0xD0,   0xA8,   0x74,   0x02 
.db 0x87,   0x9C,   0xDC,   0xA2,   0xB9,   0xBA,   0x09,   0x14 
.db 0xCC,   0x7B,   0xAE,   0x70,   0xD7,   0xC0,   0xE5,   0xD8 
.db 0x83,   0x7C,   0x07,   0xF8,   0x71,   0x48,   0x0D,   0x78 
.db 0x1C,   0xAA,   0x97,   0xA0,   0x23,   0xBC,   0x73,   0x1E 
.db 0xF2,   0x93,   0x10,   0xC4,   0x37,   0x56,   0x8D,   0x22 
.db 0x4E,   0xAD,   0x5E,   0x5C,   0x35,   0x9A,   0xED,   0xBE 
.db 0x84,   0xA5,   0xA3,   0xCE,   0x5B,   0x66,   0x06,   0x1A 
.db 0x5A,   0x4B,   0x8A,   0xCD,   0x64,   0xEA,   0x17,   0xC8 
.db 0x8F,   0x8E,   0x88,   0xEB,   0x21,   0x68,   0x4A,   0x32 
.db 0x6C,   0xCB,   0x25,   0x40,   0xFA,   0x80,   0xE1,   0x00

.org 0x580
.db "Henne's Sites [www.hoelscher-hi.de] "
.db "Revision 2.6  "
.db "GPL Freeware  "

